<?php

namespace App\Models\Concerns;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\MessageBag;

trait ApiResponseTrait
{
    /**
     * Send a unified success response
     */
    public function successResponse($data = null, string $message = 'تم بنجاح', int $code = 200): JsonResponse
    {
        $response = [
            'status'  => true,
            'type'    => 'success',
            'message' => __($message),
            'data'    => $data instanceof LengthAwarePaginator ? $data->items() : $data,
            'meta'    => [
                'timestamp' => now()->toDateTimeString(),
            ],
        ];

        if ($data instanceof LengthAwarePaginator) {
            $response['pagination'] = [
                'current_page' => $data->currentPage(),
                'last_page'    => $data->lastPage(),
                'per_page'     => $data->perPage(),
                'total'        => $data->total(),
            ];
        }

        return response()->json($response, $code);
    }

    /**
     * Send an error response
     */
    public function errorResponse(string $message = 'حدث خطأ ما', int $code = 500): JsonResponse
    {
        Log::error('API Error: ' . $message);

        return response()->json([
            'status'  => false,
            'type'    => 'error',
            'message' => __($message),
            'data'    => null,
            'meta'    => [
                'timestamp' => now()->toDateTimeString(),
            ],
        ], $code);
    }

    /**
     * Not Found Response
     */
    public function notFoundResponse(string $message = 'لم يتم العثور على البيانات المطلوبة', int $code = 404): JsonResponse
    {
        return response()->json([
            'status'  => false,
            'type'    => 'not_found',
            'message' => __($message),
            'data'    => null,
            'meta'    => [
                'timestamp' => now()->toDateTimeString(),
            ],
        ], $code);
    }

    /**
     * Unauthorized Response
     */
    public function unauthorizedResponse(string $message = 'غير مصرح لك', int $code = 401): JsonResponse
    {
        return response()->json([
            'status'  => false,
            'type'    => 'unauthorized',
            'message' => __($message),
            'data'    => null,
            'meta'    => [
                'timestamp' => now()->toDateTimeString(),
            ],
        ], $code);
    }

    /**
     * Validation Error Response
     */
    public function validationErrorResponse($errors, string $message = 'خطأ في البيانات المرسلة', int $code = 422): JsonResponse
    {
        if ($errors instanceof MessageBag) {
            $errors = $errors->toArray();
        }

        Log::warning('Validation Error', $errors);

        return response()->json([
            'status'  => false,
            'type'    => 'validation_error',
            'message' => __($message),
            'errors'  => $errors,
            'meta'    => [
                'timestamp' => now()->toDateTimeString(),
            ],
        ], $code);
    }
}
